<?php

namespace App\Http\Controllers\Website;

use App\Enums\Order\OrderStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Website\CheckoutRequest;
use App\Models\Order;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Stripe\Checkout\Session;
use Stripe\Stripe;

class CheckoutController extends Controller
{
    const STRIP_STATUS_PAID = "paid";
    const STRIP_STATUS_UNPAID = "unpaid";
    public function __construct()
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
    }

    public function checkout(CheckoutRequest $request): \Illuminate\Foundation\Application|Redirector|RedirectResponse|Application
    {
        try {
            DB::beginTransaction();
            $user = auth()->user();


            $lineItems = [];
            $model = $request->product_model;
            $product = $model::where('slug', $request->product_slug)->first();
            if (!$product) {
                return redirect()->back();
            }

            $total_amount = ($product->sale_price ?? $product->price) * 100;
            $lineItems[] = [
                'price_data' => [
                    'currency' => env('CASHIER_CURRENCY'),
                    'unit_amount' => $total_amount,
                    'product_data' => [
                        'name' => $product->name_en,
                    ],
                ],
                'quantity' => 1,
            ];

            $order = Order::create([
                'user_id' => $user->id,
                'status' => OrderStatus::PENDING->value,
                'amount' => $total_amount,
                'productable_id' => $product->id,
                'productable_type' => $model,
            ]);

            $checkoutSession = Session::create([
                'payment_method_types' => ['card'],
                'line_items' => $lineItems,
                'customer_email' => $user->email,
                'mode' => 'payment',
                'success_url' => env('APP_URL') . "/checkout-success?session_id={CHECKOUT_SESSION_ID}",
                'cancel_url' => env('APP_URL') . "/checkout-cancel?session_id={CHECKOUT_SESSION_ID}", //TODO handle cancel to order
                'metadata' => [
                    'product_model' => $model,
                    'product_id' => $product->id,
                    'user_id' => $user->id,
                    'order_id' => $order->id,
                ],
            ]);
            DB::commit();
            return redirect()->away($checkoutSession->url);
        } catch (Exception $exception) {
            DB::rollBack();
            Log::error('checkout payment generate session', ['exception' => $exception]);
            toast('Your Order failed!', 'error')->autoClose(9000);
            return redirect(route('home'));
        }

    }

    public function checkoutSuccess(Request $request)
    {
        try {
            $sessionId = $request->query('session_id');

            $checkoutSession = Session::retrieve($sessionId);

            Log::info('checkout redirection SUCCESS session Retrieve', ['checkoutSession' => $checkoutSession]);

            if ($checkoutSession->payment_status == self::STRIP_STATUS_PAID) {
                $order = Order::find($checkoutSession->metadata['order_id']);

                if (!$order) {
                    Log::error('checkout redirection exception order not found', ['order_id' => $checkoutSession->metadata['order_id']]);
                    toast('Your Order failed!', 'error')->autoClose(10000);
                    return redirect(route('home'));
                }

                $order->update([
                    'session_id' => $checkoutSession->id,
                    'status' => OrderStatus::PAID->value,
                    'transaction_json' => $checkoutSession,
                ]);

                toast('Your Order as been submitted!', 'success')->autoClose(9000);
                return redirect(route('home'));
            }

            toast('Your Order failed!', 'error')->autoClose(9000);
            return redirect(route('home'));
        } catch (Exception $exception) {
            Log::error('checkout payment redirect success function exception', ['exception' => $exception]);
            toast('Your Order failed!', 'error')->autoClose(9000);
            return redirect(route('home'));
        }


    }

    public function checkoutCancel(Request $request)
    {
        $sessionId = $request->query('session_id');

        $checkoutSession = Session::retrieve($sessionId);

        Log::info('checkout redirection FAilED session Retrieve', ['checkoutSession' => $checkoutSession]);

        if ($checkoutSession->payment_status == self::STRIP_STATUS_UNPAID) {
            $order = Order::find($checkoutSession->metadata['order_id']);

            if (!$order) {
                Log::error('checkout redirection FAilED exception order not found', ['order_id' => $checkoutSession->metadata['order_id']]);
                toast('Your Order failed!', 'error')->autoClose(10000);
                return redirect(route('home'));
            }

            $order->update([
                'session_id' => $checkoutSession->id,
                'status' => OrderStatus::Canceled->value,
                'transaction_json' => $checkoutSession,
            ]);


            toast('Your Order failed!', 'error')->autoClose(9000);
            return redirect(route('home'));
        }

    }
}
