<?php

namespace App\Repositories;

use App\DTOs\AbstractDTO;
use App\Models\Course;
use App\Repositories\Interfaces\CourseRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class CourseRepository extends Repository implements CourseRepositoryInterface
{
    public function __construct(Course $model)
    {
        $this->setModel($model);
    }

    public function get(AbstractDTO $dto): Collection|LengthAwarePaginator
    {
        $courses = $this->getModel()->active()->when($dto->getCategories(), function ($query) use ($dto){
            $query->whereIn('category_id', $dto->getCategories());
        })->orderByDesc('created_at')->newQuery();

        return $dto->getIsPaginated() ? $courses->paginate($this->getPaginationLimit()) : $courses->get();

    }

    public function getIsFeaturedCourses(AbstractDTO $dto): Collection
    {
        return $this->getModel()->active()->isFeatured()->active()->orderByDesc(
            'created_at'
        )->limit(5)->get();

    }

    public function getBySlug(string $slug): ?Model
    {
        return $this->getModel()->where('slug', $slug)->with('videos')->first();
    }

    public function getInterestedCourses(AbstractDTO $dto, string $categoryId): Collection
    {
        return $this->getModel()->active()->where('category_id', $categoryId)
            ->orderBy('created_at', 'desc')
            ->limit($dto->getLimit())
            ->get();
    }
}
